var crypto = require('crypto');
var FS = require('fs');
var Path = require('path');
var Session = require('./Session.js');
var kStatus = require('./kStatus.js');

var SessionManager = function(goEmulate, isServer, repo, config) {
    this._goEmulate = goEmulate;
    this._isServer = isServer;
    this._repo = repo;
    this._config = config;
    this._sessions = {};
    this._freeSlots = [];
    this._waitingSessions = [];
    this._activeIps = {};

    for (i = 0; i < this._config.maxSessionCount; i++) {
        this._freeSlots.push(i);
    }
};

SessionManager.prototype.getStats = function() {
    return {
        capacity: this._config.maxSessionCount,
        activeCount: this._config.maxSessionCount - this._freeSlots.length,
        waitingCount: this._waitingSessions.length
    };
};

SessionManager.prototype.getConfig = function() {
    return this._config;
};

SessionManager.prototype.setLogDir = function(dir) {
    this._logDir = dir;
}

SessionManager.prototype.createSession = function(scenarioId, ip) {
    if (this._repo.find(scenarioId) == null) {
        throw {status: kStatus.ERROR_NOT_FOUND};
    }

    var id = this.generate_id();
    var session = new Session(id, scenarioId, ip);

    if (session == null) {
        throw {status: kStatus.ERROR};
    }

    this._sessions[id] = session;
    this.addToWaiting(session);
    session.state = Session.State.WAITING;
    
    if (this._config.connectionTimeout >= 0){
        session.startConnectionTimer(this._config.connectionTimeout * 1000,
            function(session) {
                session.error = "ConnectionTimeout";
                session.log.logEvent('timeout', {reason: 'connection'});
                this.destroySession(session);
        }.bind(this));
    }

    this.checkOpening();

    return session;
};

SessionManager.prototype.closeSession = function(session) {
    session.stopConnectionTimer();

    switch (session.state) {
        case Session.State.WAITING:
            this.removeFromWaiting(session);
            session.state = Session.State.Stopped;
            this.onSessionClose(session);
            break;
        case Session.State.STARTING:
        case Session.State.STARTED:
            this.stopSession(session);
            break;
    }
};

SessionManager.prototype.restartSession = function(session, callback) {
    if (!session.isStarted()) {
        callback({status: kStatus.ERROR_STATE}, null);
        return;
    }

    session.log.logEvent('restarting');

    session.pauseMonitoring();

    session.emulator.restart(function(error, result) {
        if (error) {
            session.error = "RestartFailure";
            session.log.logEvent('error', {status: error.status});
            this.stopSession(session);
        } else {
            session.log.logEvent('restarted');
            session.resumeMonitoring();
        }
        callback(error, result);
    }.bind(this));
};

SessionManager.prototype.onSessionClose = function(session) {
    if (this._logDir) {
        var path = Path.join(this._logDir, session.id + '.json');
        var content = JSON.stringify(session.log.export(), null, 2);
        FS.writeFile(path, content, function(err) {
            // Report error?
        });
    }
};

SessionManager.prototype.destroySession = function(session) {
    if (session.id in this._sessions) {
        this.closeSession(session);

        session.log.logEvent('destroy');
        delete this._sessions[session.id];
    }
};

SessionManager.prototype.findSession = function(id) {
    var found = this._sessions[id];
    if (found) {
        return found;
    } else {
        return null;
    }
};

SessionManager.prototype.checkOpening = function() {
    if (!this.hasSlot()) {
        return;
    }

    var session = this.getNextWaiting();

    if (!session) {
        return;
    }

    this.startSession(session);
};

SessionManager.prototype.startSession = function(session) {
    var slot = this.getSlot();
    var slotParams = this.getSlotParams(slot);

    session.emulator = new this._goEmulate.Emulator();
    session.state = Session.State.STARTING;
    session.ports = slotParams.ports;
    session.slot = slot;
    session.queuePosition = 0;

    this.addIpInstance(session.ip);

    var inactiveCallback = function(session) {
        session.log.logEvent('timeout', {reason: 'inactivity'});
        this.destroySession(session);
    }.bind(this);

    var crashCallback = function(session) {
        session.log.logEvent('crash');
        this.stopSession(session);
    }.bind(this);

    session.setActivityTimeout(this._config.activityTimeout, inactiveCallback);
    session.setCrashCallback(crashCallback);

    session.log.logEvent('starting', {slot: slot});

    var startCallback = function(error, result) {
        if (error) {
            session.log.logEvent('error', {type: 'start', status: error.status})
            session.state = Session.State.STOPPED;
            session.error = "StartFailure";
            this.returnSlot(slot);
            this.removeIpInstance(session.ip);
        } else {
            session.log.logEvent('started');

            // Check state first because it may have been aborted
            if (session.state == Session.State.STARTING) {
                session.state = Session.State.STARTED;
                session.startMonitoring();
            }
        }
    }.bind(this);
    session.emulator.start(this._repo, session.scenarioId, slotParams.id,
        slotParams.ports, this._config.ipAddress, this._config.instanceIndex, this._isServer, startCallback);
};

// The session could either be STARTED or STARTING
SessionManager.prototype.stopSession = function(session) {
    session.log.logEvent('stopping');

    session.state = Session.State.STOPPING;
    session.stopMonitoring();
    this.removeIpInstance(session.ip);

    session.emulator.stop(function(error, info) {
        session.log.logEvent('stopped');

        session.emulator = null;
        session.state = Session.State.STOPPED;

        if (session.slot != null) {
            this.returnSlot(session.slot);
        }

        this.onSessionClose(session);
    }.bind(this));
};

SessionManager.prototype.addToWaiting = function(session) {
    this._waitingSessions.push(session);
    this.updateQueuePositions(this._waitingSessions.length - 1);
};

SessionManager.prototype.removeFromWaiting = function(session) {
    for (var i = 0; i < this._waitingSessions.length; i++) {
        if (this._waitingSessions[i] == session) {
            this._waitingSessions.splice(i, 1);
            this.updateQueuePositions(i);
            break;
        }
    }
};

SessionManager.prototype.getNextWaiting = function() {
    for (var i = 0; i < this._waitingSessions.length; ++i) {
        var session = this._waitingSessions[i];

        if (this.ipInstanceCount(session.ip) < this._config.maxIpSessionCount) {
            this._waitingSessions.splice(i, 1);
            this.updateQueuePositions(i);
            return session;
        } else {
            session.log.logEvent('start-delayed', {reason: 'ip-limit'});
        }
    }

    return null;
};

SessionManager.prototype.updateQueuePositions = function(startingIndex) {
    for (var i = startingIndex; i < this._waitingSessions.length; ++i) {
        this._waitingSessions[i].queuePosition = i + 1;
    }
};

SessionManager.prototype.addIpInstance = function(ip) {
    if (ip in this._activeIps) {
        ++this._activeIps[ip];
    } else {
        this._activeIps[ip] = 1;
    }
};

SessionManager.prototype.removeIpInstance = function(ip) {
    if (ip in this._activeIps) {
        --this._activeIps[ip];
        if (this._activeIps[ip] <= 0) {
            delete this._activeIps[ip];
        }
    }
};

SessionManager.prototype.ipInstanceCount = function(ip) {
    if (ip in this._activeIps) {
        return this._activeIps[ip];
    } else {
        return 0;
    }
};

SessionManager.prototype.hasSlot = function() {
    return this._freeSlots.length > 0;
};

SessionManager.prototype.getSlot = function(slot) {
    var slot = this._freeSlots[0]
    this._freeSlots.splice(0, 1);
    return slot;
};

SessionManager.prototype.returnSlot = function(slot) {
    this._freeSlots.push(slot);
    this.checkOpening();
};

SessionManager.prototype.getSlotParams = function(slot) {
    var portBase = this._config.portBase + slot * 10;
    var ports;
    if(this._isServer){
        
        ports = {
            control: portBase,
            web: portBase + 1,
            upgrade: portBase + 2,
            health: portBase + 4,
            privateData: portBase + 5,
            publicData: portBase + 6,
            udpRpc: portBase + 7
        
        };
        console.log('SessionManager: setting standard ports ' + ports.web + " " 
        + ports.upgrade + " " 
        + ports.health + " " 
        + ports.control + " "
        + ports.privateData + " " 
        + ports.publicData + " "
        + ports.udpRpc + " ");
       
    }else{
        ports = {

            control: this._config.portBase,
            web: this._config.webPort,
            upgrade: this._config.upgradePort,
            health:this._config.healthPort,
            privateData: this._config.privateDataPort,
            publicData: this._config.publicDataPort,
            udpRpc:this._config.udpRpcPort
            
        };
        console.log('SessionManager: setting custom ports ' + this._config.webPort + " " 
        + this._config.upgradePort + " " 
        + this._config.healthPort + " " 
        + this._config.portBase + " "
        + this._config.privateDataPort + " " 
        + this._config.publicDataPort + " "
        + this._config.udpRpcPort + " ");
    }
    return {
        id: this._isServer ? 1000 + slot : 0,
        ports: ports
    };
};

SessionManager.prototype.generate_id = function() {
    var key = crypto.randomBytes(24).toString('base64');

    return key.replace(/\+/g, '-').replace(/\//g, '_');
};

module.exports = SessionManager;
